import pandas as pd
import os
import json
import uuid 
from dbf import Table 

# Path absolut untuk penyimpanan file
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
UPLOAD_FOLDER = os.path.join(BASE_DIR, 'uploads')
JSON_FOLDER = os.path.join(BASE_DIR, 'converted_json')

# Pastikan folder ada
os.makedirs(UPLOAD_FOLDER, exist_ok=True)
os.makedirs(JSON_FOLDER, exist_ok=True)

def convert_file_to_json(filepath, original_filename):
    """Mengkonversi file yang diberikan ke format JSON."""
    ext = os.path.splitext(filepath)[1].lower()
    
    # Gunakan UUID untuk nama file JSON unik
    json_filename = str(uuid.uuid4()) + '.json'
    json_path = os.path.join(JSON_FOLDER, json_filename)
    json_data = None
    
    try:
        # --- Konversi Data Terstruktur (Excel, CSV) ---
        if ext in ['.xlsx', '.xls']:
            df = pd.read_excel(filepath)
            json_data = df.to_json(orient='records', indent=4)
            
        elif ext == '.csv':
            # Asumsi default encoding utf-8
            df = pd.read_csv(filepath)
            json_data = df.to_json(orient='records', indent=4)

        elif ext == '.dbf':
            # Membaca DBF
            table = Table(filepath, encoding='latin-1') 
            list_of_records = [dict(record) for record in table]
            json_data = json.dumps(list_of_records, indent=4)
            table.close()

        # --- Konversi Teks/Metadata ---
        elif ext == '.txt' or ext == '.rec':
            # Membaca TXT atau REC per baris sebagai list
            with open(filepath, 'r') as f:
                content = f.readlines()
            json_data = json.dumps({"filename": original_filename, "content_lines": [line.strip() for line in content]}, indent=4)

        elif ext == '.pdf':
            # Untuk PDF, kita hanya akan mencatat metadata dasar karena konten kompleks
            from pypdf import PdfReader
            reader = PdfReader(filepath)
            metadata = {
                "pages": len(reader.pages),
                "title": reader.metadata.title,
                "note": "Hanya metadata PDF yang diekstrak. Untuk konten (tabel/teks), diperlukan library seperti 'pdfplumber'."
            }
            json_data = json.dumps(metadata, indent=4)
            
        elif ext == '.mp4':
            # Hanya metadata sistem
            stats = os.stat(filepath)
            metadata = {
                "file_size_bytes": stats.st_size,
                "upload_time": stats.st_ctime,
                "note": "Hanya metadata sistem yang terambil. Gunakan pymediainfo untuk detail video."
            }
            json_data = json.dumps({"video_metadata": metadata}, indent=4)
            
        else:
            return None, f"Ekstensi file {ext} tidak didukung."

        # Simpan hasil konversi ke file JSON
        if json_data:
            with open(json_path, 'w') as f:
                f.write(json_data)
            return json_path, None
            
    except Exception as e:
        return None, f"Error saat konversi {ext}: {e}"

    return None, "Konversi gagal karena alasan yang tidak diketahui."