from flask import Flask, render_template, redirect, url_for, request, flash
from flask_login import LoginManager, login_user, logout_user, login_required, current_user
from werkzeug.security import generate_password_hash, check_password_hash
from werkzeug.utils import secure_filename
from functools import wraps
import os
import uuid
import json

from models import db, User, Role, ConvertedFile
from utils import convert_file_to_json, UPLOAD_FOLDER, JSON_FOLDER

# --- Inisialisasi Aplikasi ---
app = Flask(__name__)
# Ganti dengan secret key yang kuat
app.config['SECRET_KEY'] = 'kunci_super_rahasia_dan_aman_sekali'
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///' + os.path.join(os.path.dirname(os.path.abspath(__file__)), 'app.db')
app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False

db.init_app(app)
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'login'
login_manager.login_message_category = 'warning'

@login_manager.user_loader
def load_user(user_id):
    return User.query.get(int(user_id))

# --- Decorator Hak Akses ---
def role_required(role_names):
    """Membatasi akses hanya untuk peran tertentu (list string)."""
    def decorator(f):
        @wraps(f)
        def decorated_function(*args, **kwargs):
            if current_user.role.name not in role_names:
                flash("Akses ditolak: Anda tidak memiliki hak yang diperlukan.", 'danger')
                return redirect(url_for('index'))
            return f(*args, **kwargs)
        return decorated_function
    return decorator

# --- Route Inisialisasi Database (Jalankan HANYA SEKALI) ---
@app.cli.command("initdb")
def init_db_command():
    """Membuat database, peran, dan user admin awal."""
    with app.app_context():
        db.create_all()
        # 1. Buat Peran
        admin_role = Role.query.filter_by(name='Admin').first() or Role(name='Admin')
        uploader_role = Role.query.filter_by(name='Uploader').first() or Role(name='Uploader')
        user_role = Role.query.filter_by(name='User').first() or Role(name='User')

        db.session.add_all([admin_role, uploader_role, user_role])
        db.session.commit()

        # 2. Buat Admin Awal (Ganti sandi ini!)
        if not User.query.filter_by(username='admin').first():
            #hashed_pwd = generate_password_hash('admin_123', method='pbkdf2:sha256')
            hashed_pwd = generate_password_hash('admin_123', method='pbkdf2:sha256', salt_length=16)
            admin = User(username='admin', password=hashed_pwd, role=admin_role)
            db.session.add(admin)
            db.session.commit()
            print("Database diinisialisasi. Admin: admin/admin_123")
        else:
            print("Database dan User Admin sudah ada.")

# --- Route Autentikasi ---
@app.route('/login', methods=['GET', 'POST'])
def login():
    if current_user.is_authenticated:
        return redirect(url_for('index'))
    if request.method == 'POST':
        user = User.query.filter_by(username=request.form.get('username')).first()
        if user and check_password_hash(user.password, request.form.get('password')):
            login_user(user)
            return redirect(url_for('index'))
        flash('Username atau kata sandi salah.', 'danger')
    return render_template('login.html')

@app.route('/logout')
@login_required
def logout():
    logout_user()
    return redirect(url_for('login'))

# --- Route Utama ---
@app.route('/')
@login_required
def index():
    return render_template('index.html')

# --- Route Upload (Admin & Uploader) ---
@app.route('/upload', methods=['GET', 'POST'])
@login_required
@role_required(['Admin', 'Uploader'])
def upload_file():
    if request.method == 'POST':
        if 'file' not in request.files:
            flash('Tidak ada bagian file.', 'danger')
            return redirect(request.url)

        file = request.files['file']
        if file.filename == '':
            flash('Tidak ada file yang dipilih.', 'danger')
            return redirect(request.url)

        if file:
            original_filename = secure_filename(file.filename)
            # Simpan file asli dengan nama unik (UUID)
            unique_filename = str(uuid.uuid4()) + os.path.splitext(original_filename)[1]
            filepath = os.path.join(UPLOAD_FOLDER, unique_filename)
            file.save(filepath)

            # Konversi dan simpan JSON
            json_path, error = convert_file_to_json(filepath, original_filename)

            if json_path:
                # Simpan metadata di DB
                new_file = ConvertedFile(
                    original_filename=original_filename,
                    json_path=json_path,
                    uploader_id=current_user.id
                )
                db.session.add(new_file)
                db.session.commit()
                flash(f'File "{original_filename}" berhasil dikonversi dan disimpan.', 'success')
                # Hapus file asli setelah konversi
                os.remove(filepath)
                return redirect(url_for('results'))
            else:
                flash(f'Gagal mengkonversi file. Error: {error}', 'danger')
                # Hapus file yang gagal dikonversi
                if os.path.exists(filepath):
                    os.remove(filepath)
                return redirect(request.url)

    return render_template('upload.html')

# --- Route Hasil Konversi (Semua Peran) ---
@app.route('/results')
@login_required
def results():
    files = ConvertedFile.query.order_by(ConvertedFile.uploaded_at.desc()).all()
    return render_template('results.html', files=files)

# --- Route Tampilan JSON (Semua Peran) ---
@app.route('/view_json/<int:file_id>')
@login_required
def view_json(file_id):
    file_record = ConvertedFile.query.get_or_404(file_id)

    try:
        with open(file_record.json_path, 'r') as f:
            json_content = f.read()

        # Untuk tujuan tampilan yang rapi (prettify)
        pretty_json = json.dumps(json.loads(json_content), indent=4)

        return render_template('view_json.html', file_record=file_record, json_content=pretty_json)

    except Exception as e:
        flash(f"Gagal memuat file JSON. Error: {e}", 'danger')
        return redirect(url_for('results'))

# --- Route Manajemen Admin (Hanya Admin) ---
@app.route('/admin/users', methods=['GET', 'POST'])
@login_required
@role_required(['Admin'])
def manage_users():
    roles = Role.query.all()
    if request.method == 'POST':
        action = request.form.get('action')

        if action == 'add':
            username = request.form.get('username')
            password = request.form.get('password')
            role_id = request.form.get('role_id')

            if User.query.filter_by(username=username).first():
                flash('Username sudah ada.', 'danger')
                return redirect(url_for('manage_users'))

            #hashed_pwd = generate_password_hash(password, method='sha256')
            hashed_pwd = generate_password_hash(password, method='pbkdf2:sha256')
            new_user = User(username=username, password=hashed_pwd, role_id=role_id)
            db.session.add(new_user)
            db.session.commit()
            flash(f'User {username} berhasil ditambahkan.', 'success')

        elif action == 'change_role':
            user_id = request.form.get('user_id')
            new_role_id = request.form.get('new_role_id')

            user = User.query.get(user_id)
            if user:
                user.role_id = new_role_id
                db.session.commit()
                flash(f'Hak akses user {user.username} berhasil diubah menjadi {user.role.name}.', 'success')

        return redirect(url_for('manage_users'))

    users = User.query.all()
    return render_template('user_management.html', users=users, roles=roles)

if __name__ == '__main__':
    # Tidak perlu dijalankan di PythonAnywhere
    # app.run(debug=True)
    pass